

"use client";
import { useState } from "react";
import { ArrowLeft, Eye, EyeOff, AlertCircle, ShieldQuestion, Loader2, Mail, Smartphone, CheckCircle2 } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateIdentifier } from "@/lib/auth-validation";
import {  useRouter} from "next/navigation";
const BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL

export default function ForgotPassword() {
  const [step, setStep] = useState(1); // 1: ID, 2: OTP, 3: Choice, 4: Reset, 5: Success
  const [showPass, setShowPass] = useState(false);
  const [error, setError] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [methods, setMethods] = useState<any[]>([]);
  const router = useRouter()
  const [formData, setFormData] = useState({
    identifier: "",
    otp: "",
    password: "",
    confirmPassword: ""
  });

  // Step 1 -> 2: Validate ID and send initial OTP
  const handleInitialSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const vError = validateIdentifier(formData.identifier);
    if (vError) {
      setError(vError);
      return;
    }

    setIsLoading(true);
    setError("");
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/forgot/send-otp/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier }),
      });
      const data = await res.json()
      if (!res.ok){
        throw new Error(data.error);
      } 
      setStep(2);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // Fetch alternative methods
  const handleTryAnotherWay = async () => {
    setIsLoading(true);
    setError("");
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/reset/try-another-way/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier }),
      });
      const data = await res.json();

      if (!res.ok || !data.methods || data.methods.length === 0) {
        throw new Error("No other recovery options available for this account.");
      }
      
      setMethods(data.methods);
      setStep(3); 
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  const selectMethod = async (methodId: string) => {
    setIsLoading(true);
    setError("");
    try {
      const res = await fetch("/api/password/forgot/send-otp/", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier, methodId }),
      });
      if (!res.ok) throw new Error("Failed to send code to selected method.");
      setStep(2);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center p-4 w-full">
      <div className="w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] border border-border shadow-2xl overflow-hidden flex flex-col md:flex-row min-h-[550px]">
        
        {/* Left Side: Contextual Info */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
          <div className="space-y-6">
            <div className="text-2xl font-bold tracking-tighter">
                <span className="text-[#4285F4]">I</span><span className="text-[#EA4335]">n</span><span className="text-[#FBBC05]">d</span><span className="text-[#34A853]">z</span><span className="text-[#4285F4]">s</span>
            </div>
            <div className="space-y-3">
              <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                {step === 1 && "Account recovery"}
                {step === 2 && "Verification"}
                {step === 3 && "Other ways"}
                {step === 4 && "New password"}
                {step === 5 && "Updated"}
              </h1>
              <p className="text-muted-foreground text-lg font-light max-w-sm">
                {step === 1 && "Enter your email or phone number to find your account."}
                {step === 2 && "Enter the 6-digit code sent to your primary contact."}
                {step === 3 && "Choose another verification method from your account settings."}
                {step === 4 && "Choose a strong password you haven't used before."}
              </p>
            </div>
          </div>
          {step > 1 && step < 5 && (
            <button 
              onClick={() => setStep(step === 3 ? 2 : step - 1)} 
              className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-all"
            >
              <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
            </button>
          )}
        </div>

        {/* Right Side: Inputs */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
          <div className="w-full max-w-sm mx-auto">
            
            {/* STEP 1: Identification with Validation */}
            {step === 1 && (
              <form onSubmit={handleInitialSubmit} className="space-y-6">
                <div className="relative group">
                  <input 
                    type="text" 
                    required 
                    className={cn(
                        "w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent",
                        error && "border-destructive/50"
                    )}
                    placeholder="Email" 
                    value={formData.identifier} 
                    onChange={(e) => setFormData({...formData, identifier: e.target.value})} 
                  />
                  <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm px-1">Email or phone</label>
                </div>
                <div className="flex justify-between pt-4">
                  <button type="button" 
                  onClick={()=>router.push("/login")}
                  className="text-primary font-bold gap-2">Cancel</button>
                  <button type="submit" disabled={isLoading} className="bg-sidebar-primary text-white font-bold px-10 py-3 rounded-2xl flex items-center gap-2">
                    {isLoading && <Loader2 size={18} className="animate-spin" />}
                    Next
                  </button>
                </div>
              </form>
            )}

            {/* STEP 2: OTP */}
            {step === 2 && (
              <div className="space-y-6">
                <div className="relative">
                  <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
                  <input type="text" maxLength={6} className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none" value={formData.otp} onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })} />
                </div>
                <button type="button" onClick={handleTryAnotherWay} className="text-sidebar-primary text-sm font-semibold flex items-center gap-2 py-2 hover:underline">
                  <ShieldQuestion size={18} /> Try another way
                </button>
                <div className="flex justify-end pt-4">
                  <button onClick={() => setStep(4)} className="bg-sidebar-primary text-white font-bold px-10 py-3 rounded-2xl">Verify</button>
                </div>
              </div>
            )}

            {/* STEP 3: Selection Menu */}
            {step === 3 && (
              <div className="space-y-3 animate-in fade-in slide-in-from-right-4">
                {methods.map((method) => (
                  <button key={method.id} onClick={() => selectMethod(method.id)} className="w-full flex items-center gap-4 p-4 border-2 border-muted hover:border-sidebar-primary rounded-2xl transition-all text-left group">
                    <div className="bg-muted p-2 rounded-lg group-hover:bg-sidebar-primary/10 group-hover:text-sidebar-primary">
                      {method.type === 'email' ? <Mail size={20} /> : <Smartphone size={20} />}
                    </div>
                    <div>
                      <p className="font-semibold text-sm">Use {method.type}</p>
                      <p className="text-muted-foreground text-xs">{method.value}</p>
                    </div>
                  </button>
                ))}
              </div>
            )}

            {/* STEP 4: Password Reset (Form) */}
            {step === 4 && (
              <form onSubmit={(e) => { e.preventDefault(); setStep(5); }} className="space-y-4">
                 <div className="relative">
                    <input type={showPass ? "text" : "password"} required className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent" placeholder="New" onChange={(e) => setFormData({...formData, password: e.target.value})} />
                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card px-1">New password</label>
                </div>
                <div className="relative">
                    <input type={showPass ? "text" : "password"} required className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent" placeholder="Confirm" onChange={(e) => setFormData({...formData, confirmPassword: e.target.value})} />
                    <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card px-1">Confirm</label>
                    <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">{showPass ? <EyeOff size={20} /> : <Eye size={20} />}</button>
                </div>
                <button type="submit" className="w-full bg-sidebar-primary text-white font-bold py-4 rounded-2xl mt-4">Reset Password</button>
              </form>
            )}

            {/* STEP 5: Success (Centered) */}
            {step === 5 && (
              <div className="text-center space-y-6 py-8">
                <div className="flex justify-center"><CheckCircle2 size={64} className="text-emerald-500" /></div>
                <h2 className="text-xl font-bold">You're all set!</h2>
              </div>
            )}

            {/* Global Error Display */}
            {error && (
              <div className="mt-6 flex items-start gap-2 text-red-600 text-sm bg-destructive/5 p-4 rounded-xl border border-destructive/10 animate-shake">
                <AlertCircle size={16} className="mt-0.5 shrink-0" />
                <span>{error}</span>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}